/*
 * Copyright (C) 2019 - 2020, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include <string.h>

#include "internal.h"
#include "json_wrapper.h"

int acvp_str_match(const char *exp, const char *found, uint32_t id)
{
	if (!exp || !found)
		return 0;
	if (strncmp(exp, found, strlen(exp))) {
		logger(LOGGER_VERBOSE, LOGGER_C_ANY,
		       "Mismatch for ID %u (expected: %s, found: %s)\n",
		       id, exp, found);
		return -ENOENT;
	}

	return 0;
}

int acvp_get_verdict_json(const struct acvp_buf *verdict_buf,
			  enum acvp_test_verdict *verdict_stat)
{
	struct json_object *verdict_full = NULL, *verdict, *resobject;
	int ret;
	const char *result;
	bool test_passed;

	CKINT_LOG(acvp_req_strip_version(verdict_buf->buf, &verdict_full,
					 &verdict),
		  "JSON parser cannot parse verdict data\n");

	ret = json_get_bool(verdict, "passed", &test_passed);
	if (!ret) {
		*verdict_stat = acvp_verdict_pass;
		goto out;
	}

	/*
	 * Our verdict may contain a status information in case of an error
	 * and thus a different JSON structure.
	 */
	if (json_find_key(verdict, "results", &resobject, json_type_object))
		resobject = verdict;

	ret = json_get_string(resobject, "disposition", &result);
	if (ret < 0) {
		logger(LOGGER_WARN, LOGGER_C_ANY,
		       "JSON parser cannot find verdict data\n");
		*verdict_stat = acvp_verdict_unknown;
		ret = 0;
		goto out;
	}

	if (!strncmp(result, "passed", 6)) {
		*verdict_stat = acvp_verdict_pass;
	} else if (!strncmp(result, "unreceived", 10)) {
		*verdict_stat = acvp_verdict_unreceived;
	} else {
		*verdict_stat = acvp_verdict_fail;
	}

out:
	ACVP_JSON_PUT_NULL(verdict_full);
	return ret;
}

int acvp_get_algoinfo_json(const struct acvp_buf *buf,
			   struct acvp_test_verdict_status *verdict)
{
	struct json_object *algo_full = NULL, *algo;
	int ret;
	const char *tmp;

	CKINT_LOG(acvp_req_strip_version(buf->buf, &algo_full,
					 &algo),
		  "JSON parser cannot parse verdict data\n");

	CKINT(json_get_string(algo, "algorithm", &tmp));
	CKINT(acvp_duplicate(&verdict->cipher_name, tmp));

	ret = json_get_string(algo, "mode", &tmp);
	if (!ret) {
		CKINT(acvp_duplicate(&verdict->cipher_mode, tmp));
	} else {
		ret = 0;
	}

out:
	ACVP_JSON_PUT_NULL(algo_full);
	return ret;
}
