/*
 * Non-physical true random number generator based on timing jitter - DebugFS
 *
 * Copyright Stephan Mueller <smueller@chronox.de>, 2016
 *
 * License
 * =======
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU General Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

/*
 * This code tests statistical properties of the Jitter RNG by reading out
 * the following files which are stored in DebugFS under the jitterentropy-dbg/
 * directory:
 *
 * stat-fold: The returned data contains numbers in two columns, the first
 *	      column contains the timing variations for high boundary and
 *	      the second column for the low boundary of the entropy estimation.
 *	      For details, see [1] chapter 5 and appendix F.
 *
 * timer:     The data returned here shows whether the time stamp exhibits
 *	      a high-resolution and that small variations are seen. For details,
 *	      see [1] section 2.2.
 *
 * [1] http://www.chronox.de/jent/doc/CPU-Jitter-NPTRNG.pdf
 *
 *
 * Test execution:
 *
 * 1. Add the marked code below to jitterentropy.c and recompile the kernel.
 *
 * 2. Compile this module and load it into the kernel.
 *
 * 3. cat or dd of the aforementioned DebugFS files and analyze the returned
 *    data.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <asm/uaccess.h>
#include <linux/string.h>

#include <linux/debugfs.h>

struct rand_data;
extern struct rand_data *jent_entropy_collector_alloc(unsigned int osr,
						      unsigned int flags);
extern void jent_entropy_collector_free(struct rand_data *entropy_collector);
extern u64 jent_fold_var_stat(struct rand_data *ec, unsigned int min);

/* Add the following code to jitterentropy.c at the end */
#if 0
/*
 * Statistical test: return the time duration for the folding operation. If min
 * is set, perform the given number of foldings. Otherwise, allow the
 * loop count shuffling to define the number of foldings.
 */
__u64 jent_fold_var_stat(struct rand_data *ec, unsigned int min)
{
	__u64 time = 0;
	__u64 time2 = 0;
	jent_get_nstime(&time);
	jent_memaccess(ec, min);
	jent_fold_time(ec, time, min);
	jent_get_nstime(&time2);
	return ((time2 - time));
}
#include <linux/module.h>
EXPORT_SYMBOL(jent_entropy_collector_alloc);
EXPORT_SYMBOL(jent_entropy_collector_free);
EXPORT_SYMBOL(jent_fold_var_stat);
#endif

/***********************************************
 * The interface
 ***********************************************/

/* debugfs interface */
struct jent_debugfs {
	struct dentry *jent_debugfs_root; /* /sys/kernel/debug/jent */
	struct dentry *jent_debugfs_statfold; /* /sys/kernel/debug/jent/stat-fold */
	struct dentry *jent_debugfs_timer; /* /sys/kernel/debug/jent/timer */
};
static struct jent_debugfs jent_debugfs;

static ssize_t jent_debugfs_statfold_read(struct file *file, char __user *buf,
					  size_t nbytes, loff_t *ppos)
{
	char *out;
	ssize_t total = 0;
	loff_t pos = *ppos;
	ssize_t len = 0;
	struct rand_data *ec = NULL;

	ec = jent_entropy_collector_alloc(1, 0);
	if (NULL == ec)
		return -ENOMEM;

	while (0 < nbytes) {
		__u64 duration = 0;
		__u64 duration_min = 0;

		/* get data */
		duration = jent_fold_var_stat(ec, 0);
		duration_min = jent_fold_var_stat(ec, 1);

		/* feed it to user space */
		out = kasprintf(GFP_KERNEL, "%llu %llu\n", duration, duration_min);
		if (!out) {
			len = -ENOMEM;
			break;
		}
		len = strlen(out);
		/*
		 * here we potentially discard the information from one
		 * generated round -- dd will cry, but we do not care
		 */
		if (nbytes < len)
			break;
		if (copy_to_user(buf + pos + total, out, len)) {
			len = -EFAULT;
			break;
		}
		nbytes -= len;
		total += len;
		kfree(out);
	}
	jent_entropy_collector_free(ec);
	return ((0 > len) ? len : total);
}

static struct file_operations jent_statfold_fops = {
	.owner = THIS_MODULE,
	.read = jent_debugfs_statfold_read,
};

static ssize_t jent_debugfs_timer_read(struct file *file,
				       char __user *buf,
				       size_t nbytes, loff_t *ppos)
{
	char *out;
	ssize_t total = 0;
	loff_t pos = *ppos;
	size_t len = 0;

	while (0 < nbytes) {
		__u64 time, time2;
		/* get data */
		time = time2 = 0;

		//time = random_get_entropy();
		//time2 = random_get_entropy();
		time = ktime_get_ns();
		time2 = ktime_get_ns();

		/* feed it to user space */
		out = kasprintf(GFP_KERNEL, "%lld\n", (time2 - time));
		if (!out) {
			len = -ENOMEM;
			break;
		}
		len = strlen(out);
		/*
		 * here we potentially discard the information from one
		 * generated round -- dd will cry, but we do not care
		 */
		if (nbytes < len)
			break;
		if (copy_to_user(buf + pos + total, out, len)) {
			len = -EFAULT;
			break;
		}
		nbytes -= len;
		total += len;
		kfree(out);
	}

	return ((0 > len) ? len : total);
}

static struct file_operations jent_timer_fops = {
	.owner = THIS_MODULE,
	.read = jent_debugfs_timer_read,
};

/****************************************************************
 * initialization of debugfs interfaces
 ****************************************************************/

int __init jent_dbg_init(void)
{
	int ret = -EINVAL;
	jent_debugfs.jent_debugfs_statfold = NULL;
	jent_debugfs.jent_debugfs_timer = NULL;

	/* instantiate the debugfs interfaces */
	jent_debugfs.jent_debugfs_root =
		debugfs_create_dir(KBUILD_MODNAME, NULL);
	if (IS_ERR(jent_debugfs.jent_debugfs_root)) {
		printk(KBUILD_MODNAME": initialization of debugfs directory failed\n");
		return ret;
	}
	jent_debugfs.jent_debugfs_statfold =
		debugfs_create_file("stat-fold", S_IRUGO,
		jent_debugfs.jent_debugfs_root,
		NULL, &jent_statfold_fops);
	if (IS_ERR(jent_debugfs.jent_debugfs_statfold)) {
		printk(KBUILD_MODNAME": initialization of stat-fold file failed\n");
		goto cleandir;
	}
	jent_debugfs.jent_debugfs_timer =
		debugfs_create_file("timer", S_IRUGO,
		jent_debugfs.jent_debugfs_root,
		NULL, &jent_timer_fops);
	if (IS_ERR(jent_debugfs.jent_debugfs_timer)) {
		printk(KBUILD_MODNAME": initialization of timer file failed\n");
		goto cleandir;
	}
	return 0;

cleandir:
	debugfs_remove_recursive(jent_debugfs.jent_debugfs_root);

	return ret;
}

void __exit jent_dbg_exit(void)
{
	debugfs_remove_recursive(jent_debugfs.jent_debugfs_root);
}

module_init(jent_dbg_init);
module_exit(jent_dbg_exit);

MODULE_LICENSE("Dual BSD/GPL");
MODULE_AUTHOR("Stephan Mueller <smueller@chronox.de>");
MODULE_DESCRIPTION("Test code for CPU Jitter RNG");
