/*
 * Non-physical true random number generator based on timing jitter.
 *
 * Copyright Stephan Mueller <smueller@chronox.de>, 2013
 *
 * Design
 * ======
 *
 * See documentation in doc/ folder.
 *
 * Interface
 * =========
 *
 * See documentation in doc/ folder.
 *
 * License
 * =======
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU General Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "jitterentropy.h"

#ifndef CRYPTO_CPU_JITTERENTROPY_STAT
 /* only check optimization in a compilation for real work */
 #ifdef __OPTIMIZE__
  #error "The CPU Jitter random number generator must not be compiled with optimizations. See documentation. Use the compiler switch -O0 for compiling jitterentropy-base.c."
 #endif
#endif

/*
 * Update of the entropy collector / fold loop count used for the next round of
 * an entropy collection.
 *
 * Input:
 * @bits is the number of low bits of the timer to consider
 * @min is the number of bits we shift the timer value to the right at
 * 	the end to make sure we have a guaranteed minimum value
 * 
 * Return:
 * Newly calculated loop counter
 */
static unsigned int jent_loop_shuffle(unsigned int bits, unsigned int min)
{
	__u64 time = 0;
	jent_get_nstime(&time);

	/* we take the low bit of the timer which implies any
	 * value between 0 and 2^(bits + 1) - 1 */
	time = time << (64 - bits);
	time = time >> (64 - bits);

	/* We add a lower boundary value to ensure we have a minimum
	 * RNG loop count. */
	return (time + (1<<min));
}

static unsigned int jent_loop_cnt_shuffle(void)
{
/* Number of low bits of timer that are used to determine the next
 * RNG loop counter */
#define MAX_LOOP_COUNT_BIT 10
#define MIN_LOOP_COUNT_BIT 8
	return jent_loop_shuffle(MAX_LOOP_COUNT_BIT, MIN_LOOP_COUNT_BIT);
}

static unsigned int jent_loop_fold_shuffle(void)
{
/* Number of low bits of timer that are used to determine the next
 * folding loop counter */
#define MAX_FOLD_LOOP_BIT 4
#define MIN_FOLD_LOOP_BIT 0
	return jent_loop_shuffle(MAX_FOLD_LOOP_BIT, MIN_FOLD_LOOP_BIT);
}

/*
 * This function folds the time into TIME_ENTROPY_BITS bits by iterating through the
 * DATA_SIZE_BITS bit time value as follows: assume our time value
 * is 0xaaabbbcccddd, TIME_ENTROPY_BITS is 3
 * 1st loop, 1st shift generates 0xddd000000000
 * 1st loop, 2nd shift generates 0x000000000ddd
 * 2nd loop, 1st shift generates 0xcccddd000000
 * 2nd loop, 2nd shift generates 0x000000000ccc
 * 3rd loop, 1st shift generates 0xbbbcccddd000
 * 3rd loop, 2nd shift generates 0x000000000bbb
 * 4th loop, 1st shift generates 0xaaabbbcccddd
 * 4th loop, 2nd shift generates 0x000000000aaa
 * Now, the values at the end of the 2nd shifts are XORed together.
 * Note, the loop only performs (DATA_SIZE_BITS / TIME_SIZE) iterations. If the
 * division is not complete, it takes the lower bound (e.g. 64 / 3 would result
 * 21). Thus, the upmost bits that are less than TIME_SIZE in size (which are
 * assumed to have no entropy to begin with) are discarded.
 *
 * Input:
 * @time time stamp to be folded
 * @loop_cnt if a value not equal to 0 is set, use the given value as number of
 *	     loops to perform the folding
 *
 * Output:
 * @folded result of folding operation
 *
 * Return:
 * Number of loops the folding operation is performed
 */
static unsigned int jent_fold_time(__u64 time, __u64 *folded, unsigned int loop_cnt)
{
	int i, j;
	__u64 new = 0;
	unsigned int fold_loop_cnt = jent_loop_fold_shuffle();

	/* testing purposes -- allow test app to set the counter, not
	 * needed during runtime */
	if(loop_cnt)
		fold_loop_cnt = loop_cnt;
	for(j = 0; j < fold_loop_cnt; j++)
	{
		new = 0;
		for(i = 1; (DATA_SIZE_BITS / TIME_ENTROPY_BITS) >= i; i++)
		{
			__u64 tmp = time << (DATA_SIZE_BITS - (TIME_ENTROPY_BITS * i));
			tmp = tmp >> (DATA_SIZE_BITS - TIME_ENTROPY_BITS);
			new ^= tmp;
		}
	}
	*folded = new;
	return fold_loop_cnt;
}

/*
 * This is the heart of the entropy generation: calculate time deltas and
 * use the CPU jitter in the time deltas.
 *
 * Function fills rand_data->data
 *
 * Input:
 * @entropy_collector Reference to entropy collector
 *
 * Return:
 * Number of loops the entropy collection is performed.
 */
static unsigned int jent_gen_entropy(struct rand_data *entropy_collector)
{
	unsigned int k;
	unsigned int collector_loop_cnt = jent_loop_cnt_shuffle();

	/* the 0-th loop is for priming prev_time - thus we have
	 * one additional loop */
	for (k = 0; k <= collector_loop_cnt; k++)
	{
		__u64 time = 0;
		__u64 delta = 0;
		__u64 data = 0;
		__u64 prev_data = entropy_collector->data;

		jent_get_nstime(&time);
		delta = time - entropy_collector->prev_time;
		entropy_collector->prev_time = time;

		jent_fold_time(delta, &data, 0);
		/* priming of the ->prev_time value in first loop iteration */
		if(!k)
			continue;

		entropy_collector->data ^= data;
		entropy_collector->data = rol64(entropy_collector->data, 
						TIME_ENTROPY_BITS);

		/* statistics testing only */
		jent_bit_count(entropy_collector, prev_data);
	}
	return collector_loop_cnt;
}

/* the continuous test required by FIPS 140-2 -- the function automatically
 * primes the test if needed.
 *
 * Return:
 * 0 if FIPS test passed
 * < 0 if FIPS test failed
 */
static int jent_fips_test(struct rand_data *entropy_collector)
{
	if(!jent_fips_enabled())
		return 0;

	/* shall we somehow allow the caller to reset that? Probably
	 * not, because the caller can deallocate the entropy collector
	 * instance and set up a new one. */
	if(entropy_collector->fips_fail)
		return -1;

	/* prime the FIPS test */
	if(!entropy_collector->old_data)
	{
		entropy_collector->old_data = entropy_collector->data;
		jent_gen_entropy(entropy_collector);
	}

	if(entropy_collector->data == entropy_collector->old_data)
	{
		entropy_collector->fips_fail = 1;
		return -1;
	}

	entropy_collector->old_data = entropy_collector->data;

	return 0;
}

/*
 * Entry function: Obtain entropy for the caller.
 *
 * This function invokes the entropy gathering logic as often to generate
 * as many bytes as requested by the caller. The entropy gathering logic
 * creates 64 bit per invocation.
 *
 * This function truncates the last 64 bit entropy value output to the exact
 * size specified by the caller.
 *
 * @data: pointer to buffer for storing random data -- buffer must already
 *        exist
 * @len: size of the buffer, specifying also the requested number of random
 *       in bytes
 *
 * return: number of bytes returned when request is fulfilled or an error
 * 
 * The following error codes can occur:
 * 	-1	FIPS 140-2 continuous self test failed
 * 	-2	entropy_collector is NULL
 */
int jent_read_entropy(struct rand_data *entropy_collector,
		      char *data, size_t len)
{
	char *p = data;
	int ret = 0;
	size_t orig_len = len;

	if(!entropy_collector)
		return -2;

	while (0 < len)
	{
		size_t tocopy;
		jent_gen_entropy(entropy_collector);
		ret = jent_fips_test(entropy_collector);
		if(0 > ret)
			return ret;

		if(DATA_SIZE_BITS < len)
			tocopy = DATA_SIZE_BITS;
		else
			tocopy = len;
		memcpy(p, &entropy_collector->data, tocopy);

		len -= tocopy;
		p += tocopy;
	}

	/* To be on the safe side, we generate one more round of entropy
	 * which we do not give out to the caller. That round shall ensure
	 * that in case the calling application crashes, memory dumps, pages
	 * out, or due to the CPU Jitter RNG lingering in memory for long
	 * time without being moved and an attacker cracks the application,
	 * all he reads in the entropy pool is a value that is NEVER EVER
	 * being used for anything. Thus, he does NOT see the previous value
	 * that was returned to the caller for cryptographic purposes.
	 */
	/* If we use secured memory, do not use that precaution as the secure
	 * memory protects the entropy pool. Moreover, note that using this
	 * call reduces the speed of the RNG by up to half */
#ifndef CRYPTO_CPU_JITTERENTROPY_SECURE_MEMORY
	jent_gen_entropy(entropy_collector);
#endif

	return orig_len;
}

/***************************************************************************
 * End of entropy core logic
 ***************************************************************************/

/***************************************************************************
 * Initialization logic
 ***************************************************************************/

struct rand_data *jent_entropy_collector_alloc(void)
{
	struct rand_data *entropy_collector;

	entropy_collector = jent_malloc(sizeof(struct rand_data));
	if(!entropy_collector)
		return NULL;

	/**********************************************************/
	/* We are ready to roll: Initialize the entropy collector */
	memset(entropy_collector, 0, sizeof(struct rand_data));

	/* fill the data pad with non-zero values */
	jent_gen_entropy(entropy_collector);

	/* initialize the FIPS 140-2 continuous test if needed */
	jent_fips_test(entropy_collector);

	return entropy_collector;
}

void jent_entropy_collector_free(struct rand_data *entropy_collector)
{
	memset(entropy_collector, 0, sizeof(struct rand_data));
	jent_free(entropy_collector, sizeof(struct rand_data));
	entropy_collector = NULL;
}

int jent_entropy_init(void)
{

	int i;
	__u64 delta_sum = 0;
	__u64 old_delta = 0;
	int time_backwards = 0;
	int count_var = 0;
	int count_mod = 0;

	/* We could perform statistical tests here, but the problem is
	 * that we only have a few loop counts to do testing. These
	 * loop counts may show some slight skew and we produce
	 * false positives.
	 *
	 * Moreover, only old systems show potentially problematic
	 * jitter entropy that could potentially be caught here. But
	 * the RNG is intended for hardware that is available or widely
	 * used, but not old systems that are long out of favor. Thus,
	 * no statistical tests.
	 */

	/* number of loops for the entropy collection depends on the size of
	 * the random number and the size of the folded value. We want to
	 * ensure that we pass over each bit of the random value once with the
	 * folded value.  E.g. if we have a random value of 64 bits and 2 bits
	 * of folded size, we need 32 entropy collection loops. If the random
	 * value size is not divisible by the folded value size, we have as
	 * many loops to cover each random number value bit at least once. E.g.
	 * 64 bits random value size and the folded value is 3 bits, we need 22
	 * loops to cover the 64 bits at least once. */
#define ENT_LOOP ((DATA_SIZE_BITS - 1) / TIME_ENTROPY_BITS + 1)
	/* Make sure that there is no programming error -- we must at least
	 * cover the entire random number once */
	if(ENT_LOOP > 1<<MIN_LOOP_COUNT_BIT)
		return EPROGERR;

	/* We could add a check for system capabilities such as clock_getres or
	 * check for CONFIG_X86_TSC, but it does not make much sense as the
	 * following sanity checks verify that we have a high-resolution
	 * timer. */
	/* TESTLOOPCOUNT needs some loops to identify edge systems. 100 is
	 * definitely too little. */
#define TESTLOOPCOUNT 300
#define CLEARCACHE 100
	for(i = 0; (TESTLOOPCOUNT + CLEARCACHE) > i; i++)
	{
		__u64 time = 0;
		__u64 time2 = 0;
		__u64 folded = 0;
		__u64 delta = 0;
		
		jent_get_nstime(&time);
		jent_fold_time(time, &folded, 1<<MIN_FOLD_LOOP_BIT);
		jent_get_nstime(&time2);

		/* test whether timer works */
		if(!time || !time2)
			return ENOTIME;
		delta = time2 - time;
		/* test whether timer is fine grained enough to provide
		 * delta even when called shortly after each other -- this
		 * implies that we also have a high resolution timer */
		if(!delta)
			return ECOARSETIME;
		/* TIME_ENTROPY_BITS states the absolute minimum entropy we
		 * assume the time variances have. As we also check for
		 * delta of deltas, we ensure that there is a varying delta
		 * value, preventing identical time spans */
		if(TIME_ENTROPY_BITS > delta)
			return EMINVARIATION;

		/* up to here we did not modify any variable that will be evaluated
		 * later, but we already performed some work. Thus we already
		 * have had an impact on the caches, branch prediction, etc. with
		 * the goal to clear it to get the worst case measurements.
		 */
		if(CLEARCACHE > i)
			continue;

		/* test whether we have an increasing timer */
		if(!(time2 > time))
			time_backwards++;

		if(!(delta % 10))
			count_mod++;

		/* ensure that we have a varying delta timer which is necessary
		 * for the calculation of entropy -- perform this check
		 * only after the first loop is executed as we need to prime
		 * the old_data value */
		if(i)
		{
			if(delta != old_delta)
				count_var++;
			if(delta > old_delta)
				delta_sum += (delta - old_delta);
			else
				delta_sum += (old_delta - delta);
		}
		old_delta = delta;

	}

	/* we allow up to three times the time running backwards.
	 * CLOCK_REALTIME is affected by adjtime and NTP operations. Thus,
	 * if such an operation just happens to interfere with our test, it
	 * should not fail. The value of 3 should cover the NTP case being
	 * performed during our test run. */
	if(3 < time_backwards)
		return ENOMONOTONIC;
	/* Error if the time variances are always identical */
	if(!delta_sum)
		return EVARVAR;

	/* Variations of deltas of time must on average be larger
	 * than TIME_ENTROPY_BITS to ensure the entropy estimation
	 * implied with TIME_ENTROPY_BITS is preserved */
	if(!(delta_sum / TESTLOOPCOUNT) > TIME_ENTROPY_BITS)
		return EMINVARVAR;

	/* Ensure that we have variations in the time stamp below 10 for at least
	 * 10% of all checks -- on some platforms, the counter increments in
	 * multiples of 10, but not always */
	if((TESTLOOPCOUNT/10 * 9) < count_mod)
		return ECOARSETIME;

	return 0;
}

/***************************************************************************
 * Statistical test logic not compiled for regular operation
 ***************************************************************************/

#ifdef CRYPTO_CPU_JITTERENTROPY_STAT
/* Statistical tests: invoke the entropy collector and sample time results
 * for it, the random data is never returned - every call to this function
 * generates one random number
 * This function is only meant for statistical analysis purposes and not
 * for general use
 */
void jent_gen_entropy_stat(struct rand_data *entropy_collector,
	       		   struct entropy_stat *stat)
{
	int cnt;
	/* caller is allowed to set the entropy collection loop to a fixed
	 * value -- we still call shuffle for the time measurements */
	jent_init_statistic(entropy_collector);
	cnt = jent_gen_entropy(entropy_collector);
	jent_calc_statistic(entropy_collector, stat, cnt);
}

/* Statistical test: obtain the distribution of the folded time value from
 * jent_fold_time */
void jent_fold_time_stat(__u64 *fold, unsigned int *loop_cnt)
{
	__u64 time = 0;
	__u64 time2 = 0;
	__u64 folded = 0;
	jent_get_nstime(&time);
	/* implement the priming logic */
	jent_fold_time(time, &folded, 0);
	jent_get_nstime(&time2);
	time2 = time2 - time;
	*loop_cnt = jent_fold_time(time2, &folded, 0);
	*fold = folded;

}

/* Statistical test: return the time duration for the folding operation. If min is
 * set, perform the absolute minimum number of foldings. Otherwise, allow the
 * loop count shuffling to define the mumber of foldings. */
__u64 jent_fold_var_stat(int min)
{
	__u64 time = 0;
	__u64 time2 = 0;
	__u64 folded = 0;
	jent_get_nstime(&time);
	if(min)
		jent_fold_time(time, &folded, 1<<MIN_FOLD_LOOP_BIT);
	else
		jent_fold_time(time, &folded, 0);
	jent_get_nstime(&time2);
	return ((time2 - time));
}
#endif /* CRYPTO_CPU_JITTERENTROPY_STAT */


